package ptft;

/*
 * JPtftCellularAutomata.java
 *
 * Written by Will Braynen
 * Group for Logic and Formal Semantics, SUNY Stony Brook (www.ptft.org)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Created on May 11, 2004, 10:34 PM
 */


import java.awt.*;
// import ptft.cell.*;

/**
 * A graph paper grid populated with <code>JCell</code> objects which play
 * localized cellular automata games. Each cell can have one of eight
 * possible Markov strategies specifying how to respond to neighboring cells,
 * plus the polymorphic PTFT strategy:
 * <p>
 * 0 = 000 = AllD (always defect)<br>
 * 1 = 001<br>
 * 2 = 010 = STFT (suspicious tit-for-tat: D-then-TFT)<br>
 * 3 = 011 = D-then-AllC (defect only the first time)<br>
 * 4 = 100 = C-then-AllD (cooperate only the first time)<br>
 * 5 = 101<br>
 * 6 = 110 = TFT (tit-for-tat: C-then-TFT)<br>
 * 7 = 111 = AllC (always cooperate)<br>
 * 8 = PTFT (TFT with one's own kind, AllD with others)
 * <p>
 * A cell with a PTFT (prejudiced tit-for-tat)
 * strategy plays TFT (tit-for-tat) against its own kind -- neighboring
 * cells that have the same ethnicity -- but always defects against anyone
 * of a different ethnicity.
 * <p>
 * Calling <code>playRound</code> evolves the cellular automata one gerenation
 * by having each cell play 200 games against each of its eight neighboring 
 * cells. The points from each of the eight interactions are added up and 
 * become the cell's score.  Each cell then adopts the strategy of its most 
 * successful neighbor -- the neighboring cell with the highest score.
 * The updating is instantenous; the entire array is switched to the 
 * new strategies in one shot, not cell by cell.
 *
 * @author  Will Braynen
 */
public class JPtftCellularAutomata extends JCellularAutomata {

    // fields
    public boolean m_isStrategyDisplayed = true;
    protected boolean m_isSegregated = false;
    protected int m_cellWidth = 6;
    protected CellFactory factory;
    
    protected final Color[] m_strategyColors = // redeclaring to add PTFT
    {
        Color.green,     // 0 = 000 = AllD (always defect)
        Color.cyan,      // 1 = 001
        Color.red,       // 2 = 010 = STFT (suspicious tit-for-tat: D-then-TFT)
        Color.magenta,   // 3 = 011
        Color.yellow,    // 4 = 100 = C-then-AllD (cooperate only the first time)
        Color.lightGray, // 5 = 101
        Color.gray,      // 6 = 110 = TFT (tit-for-tat: C-then-TFT)
        Color.blue,      // 7 = 111 = AllC (always cooperate)
        Color.white      // 8 = PTFT (TFT with one's own kind, AllD with others)
    };
    protected final Color[] m_ethnicColors =
    {
        Color.red,
        Color.green
    };
    

    /** Creates new form JPtftCellularAutomata */
    public JPtftCellularAutomata(int cellWidth) {
        super(cellWidth);
        m_cellWidth = cellWidth;
        //initComponents(); // DO NOT CALL THIS -- ALREADY CALLED IN PARENT CLASS
    }

    /** 
     * Adds cells to the GUI.  
     * Overrides the parent's method to add ethnicities 
     */

    // NamedColorArray strategies, NamedColorArray ethnicities ) {
    public void createCells()
    {
        factory = CellFactory.getInstance();
        
        // m_strategies = strategies;
        
        // create cells and add them to GUI
        for (int x = 0;  x < m_rows;  x++)
        {
            for (int y = 0;  y < m_columns;  y++)
            {
                JCell cell = factory.create(x,y);
                m_grid[x][y] = cell;

                // add the cell to the GUI
                innerPanel.add (cell);
            }
        }
        
    }
    
    /** Adds cells to the GUI.  Overrides the parent's method to add ethnicities */
    public void createCells( NamedColorArray strategies, NamedColorArray ethnicities ) {
        
        m_strategies = strategies;
        
        // create cells and add them to GUI
        for (int x = 0;  x < m_rows;  x++)
        {
            for (int y = 0;  y < m_columns;  y++)
            {
                JCell cell = new JPtftCell(x, y, m_cellWidth, strategies, ethnicities);
                m_grid[x][y] = cell;

                // add the cell to the GUI
                innerPanel.add (cell);
            }
        }
    } // end initCells
    
    public void restart() {
        
        super.restart();
        setRandomEthnicities();
    }

    public void setRandomEthnicities() {
        
        setRandomEthnicities( null );
    }
    
    public void setRandomEthnicities( int[] ethnicities ) {
        
        for (int x = 0;  x < m_rows;  x++)
        {
            for (int y = 0;  y < m_columns;  y++)
            {
                JPtftCell cell = (JPtftCell) m_grid[x][y]; 
                cell.setRandomEthnicity( ethnicities );
            }
        }
        m_isSegregated = false;
    } // end setRandomEthnicities
        
    public boolean isStrategyDisplayed()
    {
        return m_isStrategyDisplayed;
    }
    
    public boolean isSegregated()
    {
        return m_isSegregated;
    }
    
    public void toggleViews()
    {
        for (int x = 0;  x < m_rows;  x++)
        {
            for (int y = 0;  y < m_columns;  y++)
            {
                JPtftCell cell = (JPtftCell) m_grid[x][y];
                cell.toggleViews();
            }
        }
        m_isStrategyDisplayed = ( ! m_isStrategyDisplayed );
        repaint();
    } // end toggleViews

    public void segregate() 
    {
        for (int x = 0;  x < m_rows;  x++)
        {
            for (int y = 0;  y < m_columns;  y++)
            {
                JPtftCell cell = (JPtftCell) m_grid[x][y];

                if ( y < (m_rows / 2) )
                {
                    // if on the left half of the grid, make cells one color
                    cell.setEthnicity (0);
                }
                else
                {
                    // if on the right half of the grid, make cells another color
                    cell.setEthnicity (1);
                }
            }
        }
        m_isSegregated = true;
        repaint();
    } // end segregate
    
    public void integrate() 
    {
        setRandomEthnicities();
        m_isSegregated = false;
        repaint();
    } // end integrate
    
    protected int playNeighbor( int x, int y, Direction d ) 
    {        
        JPtftCell cell = (JPtftCell) getCell (x, y);
        JPtftCell neighbor = (JPtftCell) getNeighbor (cell, d);        
        return m_payoffs[cell.getStrategy(neighbor)][neighbor.getStrategy(cell)];
    } // end playNeighbor
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents() {//GEN-BEGIN:initComponents

        setLayout(null);

    }//GEN-END:initComponents
    
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    // End of variables declaration//GEN-END:variables
    
}
