package ptft;

/*
 * JCellularAutomata.java
 *
 * Written by Will Braynen
 * Group for Logic and Formal Semantics, SUNY Stony Brook (www.ptft.org)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Created on May 11, 2004, 4:36 PM
 */

import java.awt.*;

/**
 * A graph paper grid populated with <code>JCell</code> objects which play
 * localized cellular automata games. Each cell can have one of eight
 * possible Markov strategies specifying how to respond to neighboring cells:
 * <p>
 * 0 = 000 = AllD (always defect)<br>
 * 1 = 001<br>
 * 2 = 010 = STFT (suspicious tit-for-tat: D-then-TFT)<br>
 * 3 = 011 = D-then-AllC (defect only the first time)<br>
 * 4 = 100 = C-then-AllD (cooperate only the first time)<br>
 * 5 = 101<br>
 * 6 = 110 = TFT (tit-for-tat: C-then-TFT)<br>
 * 7 = 111 = AllC (always cooperate)
 * <p>
 * Calling <code>playRound</code> evolves the cellular automata one gerenation
 * by having each cell play 200 games against each of its eight neighboring 
 * cells. The points from each of the eight interactions are added up and 
 * become the cell's score.  Each cell then adopts the strategy of its most 
 * successful neighbor -- the neighboring cell with the highest score.
 * The updating is instantenous; the entire array is switched to the 
 * new strategies in one shot, not cell by cell.
 *
 * @author  Will Braynen
 */
public class JCellularAutomata extends JGrid implements Runnable {
    
    // fields
    protected int m_payoffs[][];
    protected Thread m_thread;
    protected int m_sleepDelay = 500; // delay between frames in milliseconds
        
    /** Creates a cellular automata grid with the following default
     * prisoner's dilemma payoff matrix:
     *
     * <table border=0 bgcolor="#E0E0E0" align=center>
     *   <tr align=center><td bgcolor="#FFFFFF">                          </td><td bgcolor="#FFFFFF"><code>  C  </code></td><td bgcolor="#FFFFFF"><code>  D  </code></td></tr>
     *   <tr align=center><td bgcolor="#FFFFFF"><code>&nbsp;C&nbsp;</code></td><td bgcolor="#FFFFFF">      (3,3)       </td><td bgcolor="#FFFFFF">      (0,5)       </td></tr>
     *   <tr align=center><td bgcolor="#FFFFFF"><code>&nbsp;D&nbsp;</code></td><td bgcolor="#FFFFFF">      (5,0)       </td><td bgcolor="#FFFFFF">      (1,1)       </td></tr>
     * </table><br>
     */
    public JCellularAutomata(int cellWidth) {
        
        super(cellWidth);
        
        //initComponents(); // DO NOT CALL THIS -- ALREADY CALLED IN PARENT CLASS
        m_payoffs = new int[8][8];
        setPayoffs (1, 5, 0, 3);
        
    } // end constructor
        
    protected int getWinningStrategy( JCell cell ) {
        
        int winningStrategy = -1; // will be returned
        int strategies[] = new int[9];
        int scores[] = new int[9];

        // The indices for arrays scores[] and strategies[] represent
        // neighboring cells numbered in the following way,with the
        // cell that's doing the surveying being in the center (#8)
        // in the grid below:
        // |---|---|---|
        // | 0 | 5 | 1 |
        // |---|---|---|
        // | 4 | 8 | 6 |
        // |---|---|---|
        // | 2 | 7 | 3 |
        // |---|---|---|
        
        // cache the scores
        scores[0] = getNeighbor (cell, Direction.NW).m_score;
        scores[1] = getNeighbor (cell, Direction.NE).m_score;
        scores[2] = getNeighbor (cell, Direction.SW).m_score;
        scores[3] = getNeighbor (cell, Direction.SE).m_score;
        scores[4] = getNeighbor (cell, Direction.W).m_score;
        scores[5] = getNeighbor (cell, Direction.N).m_score;
        scores[6] = getNeighbor (cell, Direction.E).m_score;
        scores[7] = getNeighbor (cell, Direction.S).m_score;
        scores[8] = cell.m_score;
        
        // cache the strategies
        strategies[0] = getNeighbor (cell, Direction.NW).m_strategy;
        strategies[1] = getNeighbor (cell, Direction.NE).m_strategy;
        strategies[2] = getNeighbor (cell, Direction.SW).m_strategy;
        strategies[3] = getNeighbor (cell, Direction.SE).m_strategy;
        strategies[4] = getNeighbor (cell, Direction.W).m_strategy;
        strategies[5] = getNeighbor (cell, Direction.N).m_strategy;
        strategies[6] = getNeighbor (cell, Direction.E).m_strategy;
        strategies[7] = getNeighbor (cell, Direction.S).m_strategy;
        strategies[8] = cell.m_strategy;

        
        // Start comparing with a random neighbor
        double high = 0;
        int startWithNeighbor = (int)(Math.random() * 8);
        int r;
        for (r = startWithNeighbor; r <= 8; r++)
        {
            // Have the cell adopt the strategy of its most successful neighbor
            if (scores[r] >= high) // should this be ">" instead?
            {
              high = scores[r];
              winningStrategy = strategies[r];
            }
        }
        for (r = 0; r < startWithNeighbor; r++)
        {
            // Have the cell adopt the strategy of its most successful neighbor
            if (scores[r] >= high) // should this be ">" instead?
            {
                high = scores[r];
                winningStrategy = strategies[r];
            }
        }
        // the cell's strategy is best
        if (scores[8] >= high)
        {
            winningStrategy = strategies[8];
        }
        return winningStrategy;
    } // end getWinningStrategy
    
    /** calls <code>playRound</code> with <code>repaint</code> = true */
    public void playRound() {
        playRound (true);
    }
    
    /**
     * Evolves the cellular automata one gerenation by having each cell play
     * 200 games against each of its eight neighboring cells. The points from
     * each of the eight interactions are added up and become the cell's
     * score.  Each cell then adopts the strategy of its most successful 
     * neighbor -- the neighboring cell with the highest score.
     * The updating is instantenous; the entire array is switched to the 
     * new strategies in one shot, not cell by cell.
     * 
     * @param  repaint  You might want not to repaint the grid. For instance, 
     *                  you might want to display only every tenth generation 
     *                  or perhaps display only the final state. Dropping
     *                  frames in this way might make the cellular automata
     *                  evolve faster, and so could be used as a performance
     *                  optimization technique.
     */
    public void playRound( boolean repaint ) {
        
        //
        // Have each cell play one round against each of 
        // its eight adjacent neighbors
        //
        int rows    = getTotalRows();
        int columns = getTotalColumns();
        
		for (int x = 0;  x < rows;  x++) {
			for (int y = 0;  y < columns;  y++) {
				
				int score = playCell(x, y);
				JCell oneCell = m_grid[x][y];
				
				//add code here to factor in inequality of colors
				//System.out.println("old: " + score);
				int randomPercent = oneCell.getRandomPayoffPercentage();
				score = (int)(score * randomPercent / 100);
				//System.out.println("new: " + score + " " + randomPercent + "%");
				
				oneCell.m_score = score;
			
			}
		}
        
        //
        // Darwinian survival of the fittest - let the best
        // strategy prevail!  Have each cell adopt the
        // strategy of its most successful adjacent neighbor.
        //
        for (int x = 0;  x < rows;  x++) {
            for (int y = 0;  y < columns;  y++) {
               m_grid[x][y].m_newStrategy = getWinningStrategy( getCell( x, y ) );
            }
        }
        
        // Switch to the new strategies for next round
        // - a new, "stronger" generation is born!
        for (int x = 0;  x < rows;  x++) {
            for (int y = 0;  y < columns;  y++) {
               // init / zero out the scores
               m_grid[x][y].m_score = 0;
               
               // switch to the new strategy
               m_grid[x][y].m_strategy = m_grid[x][y].m_newStrategy;

               // Shawn and Harry want to either do this:
               //  m_grid[x][y].switchToNewStrategy();
            }
        }
		
		//increment the generation count
		this.generation++;
        
        if (repaint) this.repaint();        
        
    } // end playRound
	
	        
    protected int playNeighbor( int x, int y, Direction d ) {        
        JCell cell = getCell (x, y);
        JCell neighbor = getNeighbor (cell, d);        
        return m_payoffs[cell.getStrategy()][neighbor.getStrategy()];
    }
            
    /** have a cell play all of its neighbors */
    protected int playCell( int x, int y ) {
        int selfScore = 0, Nscore, Sscore, Escore, Wscore, NWscore, NEscore, SWscore, SEscore;
        // |---|---|---|
        // | NW| N |NE |
        // |---|---|---|
        // | W |x,y| E |
        // |---|---|---|
        // | SW| S |SE |
        // |---|---|---|
        
        Nscore = playNeighbor (x, y, Direction.N);
        Sscore = playNeighbor (x, y, Direction.S);
        Escore = playNeighbor (x, y, Direction.E);
        Wscore = playNeighbor (x, y, Direction.W);
        NWscore = playNeighbor (x, y, Direction.NW);
        NEscore = playNeighbor (x, y, Direction.NE);
        SWscore = playNeighbor (x, y, Direction.SW);
        SEscore = playNeighbor (x, y, Direction.SE);
		
		
		int totalScore = selfScore + Nscore + Sscore + Escore + Wscore + NWscore + NEscore + SWscore + SEscore;
        
        return totalScore;
    }       
    
    /**
     * Set the payoffs for any symmetrical 2x2 payoff matrix.
     *
     * <table border=0 bgcolor="#E0E0E0" align=center>
     *   <tr align=center><td bgcolor="#FFFFFF">                          </td><td bgcolor="#FFFFFF"><code>&nbsp;C&nbsp;</code></td><td bgcolor="#FFFFFF"><code>&nbsp;D&nbsp;</code></td></tr>
     *   <tr align=center><td bgcolor="#FFFFFF"><code>&nbsp;C&nbsp;</code></td><td bgcolor="#FFFFFF">           cc             </td><td bgcolor="#FFFFFF">           cd             </td></tr>
     *   <tr align=center><td bgcolor="#FFFFFF"><code>&nbsp;D&nbsp;</code></td><td bgcolor="#FFFFFF">           dc             </td><td bgcolor="#FFFFFF">           dd             </td></tr>
     * </table><br>
     *
     * @param  dd  points awarded for mutual defection
     * @param  dc  points awarded for unilateral defection
     * @param  cd  points awarded for unilateral cooperation
     * @param  cc  points awarded for mutual cooperation 
     */
    public void setPayoffs( int dd, int dc, int cd, int cc ) {
        // 000
        m_payoffs[0][0] = 200 * dd;
        m_payoffs[0][1] = dd + 199 * dc;
        m_payoffs[0][2] = 200 * dd;
        m_payoffs[0][3] = dd + 199 * dc;
        m_payoffs[0][4] = dc + 199 * dd;
        m_payoffs[0][5] = 200 * dc;
        m_payoffs[0][6] = dc + 199 * dd;
        m_payoffs[0][7] = 200 * dc;

        // 001
        m_payoffs[1][0] = dd + 199 * cd;
        m_payoffs[1][1] = 100 * dd + 100 * cc;
        m_payoffs[1][2] = 50 * (cc + cd + dc + dd); // !
        m_payoffs[1][3] = dd + cc + 198 * dc;
        m_payoffs[1][4] = dc + dd + 198 * cd;
        m_payoffs[1][5] = 200 * dc;
        m_payoffs[1][6] = 50 * (cc + cd + dc + dd);
        m_payoffs[1][7] = 200 * dc;

        // 010
        m_payoffs[2][0] = 200 * dd;
        m_payoffs[2][1] = 50 * (cc + cd + dc + dd);
        m_payoffs[2][2] = 200 * dd;
        m_payoffs[2][3] = dd + dc + 198 * cc;
        m_payoffs[2][4] = dc + cd + 198 * dd;
        m_payoffs[2][5] = 50 * (cc + cd + dc + dd);
        m_payoffs[2][6] = 100 * dc + 100 * cd;
        m_payoffs[2][7] = dc + 199 * cc;

        // 011
        m_payoffs[3][0] = dd + 199 * cd;
        m_payoffs[3][1] = dd + cc + 198 * cd;
        m_payoffs[3][2] = dd + cd + 198 * cc;
        m_payoffs[3][3] = dd + 199 * cc;
        m_payoffs[3][4] = dc + 199 * cd;
        m_payoffs[3][5] = dc + cc + 198 * cd;
        m_payoffs[3][6] = dc + cd + 198 * cc;
        m_payoffs[3][7] = dc + 199 * cc;

        // 100
        m_payoffs[4][0] = cd + 199 * dd;
        m_payoffs[4][1] = cd + dd + 198 * dc;
        m_payoffs[4][2] = cd + dc + 198 * dd;
        m_payoffs[4][3] = cd + 199 * dc;
        m_payoffs[4][4] = cc + 199 * dd;
        m_payoffs[4][5] = cc + dd + 198 * dc;
        m_payoffs[4][6] = cc + dc + 198 * dd;
        m_payoffs[4][7] = cc + 199 * dc;

        // 101
        m_payoffs[5][0] = 200 * cd;
        m_payoffs[5][1] = 200 * cd;
        m_payoffs[5][2] = 50 * (cc + cd + dc + dd);
        m_payoffs[5][3] = cd + cc + 198 * dc;
        m_payoffs[5][4] = cc + dd + 198 * cd;
        m_payoffs[5][5] = 100 * cc + 100 * dd;
        m_payoffs[5][6] = 50 * (cc + cd + dc + dd);
        m_payoffs[5][7] = cc + 199 * dc;

        // 110 (TFT)
        m_payoffs[6][0] = cd + 199 * dd;
        m_payoffs[6][1] = 50 * (cc + cd + dc + dd);
        m_payoffs[6][2] = 100 * cd + 100 * dc;
        m_payoffs[6][3] = cd + dc + 198 * cc;
        m_payoffs[6][4] = cc + cd + 198 * dd;
        m_payoffs[6][5] = 50 * (cc + cd + dc + dd);
        m_payoffs[6][6] = 200 * cc;
        m_payoffs[6][7] = 200 * cc;

        // 111
        m_payoffs[7][0] = 200 * cd;
        m_payoffs[7][1] = 200 * cd;
        m_payoffs[7][2] = cd + 199 * cc;
        m_payoffs[7][3] = cd + 199 * cc;
        m_payoffs[7][4] = cc + 199 * cd;
        m_payoffs[7][5] = cc + 199 * cd;
        m_payoffs[7][6] = 200 * cc;
        m_payoffs[7][7] = 200 * cc;
    } // initPayoffs
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents() {//GEN-BEGIN:initComponents

        setLayout(null);

    }//GEN-END:initComponents


    public void restart() {
        
   		this.generation = 0;
		setRandomStrategies();

	}
    
    public void start() {
        
        if (null == m_thread)
        {
            // create a thread
            m_thread = new Thread (this, "Cellular Automata");
            m_thread.start();
        }
    }
    
    public void pause() {
        
        m_thread = null;
    }
        
    public void run() {
        
        while (null != m_thread)
        {
            playRound();
            try {
                Thread.sleep( m_sleepDelay );
            } catch (InterruptedException e) {
                // the VM doesn't want us to sleep anymore,
                // so get back to work
            }
        }
    }
    
    public boolean isRunning() {
        
        return (null != m_thread);
    }

    /**
     * @param  sleepDelayMS  sleep delay between <code>playRound</code> frames 
     *                       in milliseconds
     */
    public void setSleepDelay( int sleepDelayMS )
    {
        m_sleepDelay = sleepDelayMS;
    }
    
	public void repaintAll() {
		repaint();
	}
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    // End of variables declaration//GEN-END:variables
    
}
